/**
 * CarbonGuru Manifest V3 Service Worker - Compatibility Wrapper
 *
 * This is a thin compatibility layer that provides polyfills for APIs not available
 * in MV3 service workers, then loads the actual extension code (background.js).
 *
 * Firefox loads background.js directly as a background page.
 * Chrome loads this wrapper as a service worker, which then loads background.js.
 *
 * Result: Same business logic, different entry points.
 *
 * Key MV3 Adaptations:
 * - setInterval → chrome.alarms (survives service worker restarts)
 * - State persistence via chrome.storage.local (polling resumes on restart)
 * - One-shot alarms for sub-60s intervals (chrome.alarms minimum is 30s in dev)
 */

console.log('[CarbonGuru MV3 Wrapper] Initializing compatibility layer...');

// ==================== POLYFILLS ====================

/**
 * 1. Polyfill 'window' for code that checks typeof window
 * background.js uses this for cache detection (lines 24-27)
 */
if (typeof window === 'undefined') {
    self.window = self;
}

/**
 * 2. Wrap setInterval/clearInterval for service worker reliability
 *
 * Service workers can be terminated after ~30s of inactivity.
 *
 * Strategy:
 * - Intervals < 10s: Use native setInterval (active processing, SW stays awake)
 * - Intervals 10s-59s: Use one-shot alarms with self-rescheduling
 * - Intervals >= 60s: Use periodic alarms (chrome.alarms native support)
 *
 * All alarm-based intervals are persisted to storage.local so they can
 * resume after service worker restarts.
 */
const alarmCallbacks = new Map();
let alarmCounter = 0;
const ALARM_PREFIX = 'cg_interval_';
const STORAGE_KEY = 'cg_mv3_intervals';

const nativeSetInterval = self.setInterval.bind(self);
const nativeClearInterval = self.clearInterval.bind(self);

/**
 * Persist interval metadata to storage for SW restart recovery
 */
async function persistIntervals() {
    const intervals = {};
    for (const [name, entry] of alarmCallbacks.entries()) {
        // Only persist alarm-based intervals (not native ones)
        if (name.startsWith(ALARM_PREFIX)) {
            intervals[name] = {
                ms: entry.ms,
                callbackName: entry.callbackName || null,
                createdAt: entry.createdAt
            };
        }
    }
    try {
        await chrome.storage.local.set({ [STORAGE_KEY]: intervals });
    } catch (e) {
        console.warn('[CarbonGuru MV3 Wrapper] Failed to persist intervals:', e);
    }
}

/**
 * Schedule a one-shot alarm (for sub-60s intervals)
 */
function scheduleOneShotAlarm(alarmName, ms) {
    chrome.alarms.create(alarmName, { when: Date.now() + ms });
}

self.setInterval = (callback, ms, ...args) => {
    const id = ++alarmCounter;

    // For very short intervals (< 10s), use native setInterval
    // These are for active polling during processing when SW is awake
    if (ms < 10000) {
        const nativeId = nativeSetInterval(callback, ms, ...args);
        // Store mapping so clearInterval works
        alarmCallbacks.set(`native_${id}`, { nativeId, isNative: true });
        return id;
    }

    const alarmName = `${ALARM_PREFIX}${id}`;

    // Store callback info
    alarmCallbacks.set(alarmName, {
        callback,
        args,
        ms,
        createdAt: Date.now(),
        // Try to identify the callback for logging/debugging
        callbackName: callback.name || 'anonymous'
    });

    if (ms >= 60000) {
        // For intervals >= 60s, use periodic alarms
        const minutes = Math.max(ms / 60000, 1);
        chrome.alarms.create(alarmName, { periodInMinutes: minutes });
        console.log(`[CarbonGuru MV3 Wrapper] Created periodic alarm ${alarmName} (${minutes} min)`);
    } else {
        // For intervals 10s-59s, use one-shot alarms that reschedule
        scheduleOneShotAlarm(alarmName, ms);
        console.log(`[CarbonGuru MV3 Wrapper] Created one-shot alarm ${alarmName} (${ms}ms)`);
    }

    // Persist for SW restart recovery
    persistIntervals();

    return id;
};

self.clearInterval = (id) => {
    // Check for native interval
    const nativeKey = `native_${id}`;
    if (alarmCallbacks.has(nativeKey)) {
        const entry = alarmCallbacks.get(nativeKey);
        nativeClearInterval(entry.nativeId);
        alarmCallbacks.delete(nativeKey);
        return;
    }

    // Check for alarm-based interval
    const alarmName = `${ALARM_PREFIX}${id}`;
    if (alarmCallbacks.has(alarmName)) {
        chrome.alarms.clear(alarmName);
        alarmCallbacks.delete(alarmName);
        console.log(`[CarbonGuru MV3 Wrapper] Cleared alarm ${alarmName}`);
        persistIntervals();
        return;
    }

    // Fallback: try clearing as native interval ID directly
    nativeClearInterval(id);
};

/**
 * 2b. Wrap setTimeout/clearTimeout for service worker reliability
 *
 * setTimeout with delays >= 1000ms can fail if SW terminates.
 * Use alarms for these longer delays.
 */
const TIMEOUT_PREFIX = 'cg_timeout_';
const timeoutCallbacks = new Map();
let timeoutCounter = 0;

const nativeSetTimeout = self.setTimeout.bind(self);
const nativeClearTimeout = self.clearTimeout.bind(self);

self.setTimeout = (callback, ms, ...args) => {
    const id = ++timeoutCounter;

    // For short timeouts (< 1s), use native setTimeout
    if (ms < 1000) {
        const nativeId = nativeSetTimeout(callback, ms, ...args);
        timeoutCallbacks.set(`native_timeout_${id}`, { nativeId, isNative: true });
        return id;
    }

    const alarmName = `${TIMEOUT_PREFIX}${id}`;

    // Store callback info
    timeoutCallbacks.set(alarmName, {
        callback,
        args,
        ms,
        createdAt: Date.now(),
        callbackName: callback.name || 'anonymous'
    });

    // Create one-shot alarm
    chrome.alarms.create(alarmName, { when: Date.now() + ms });
    console.log(`[CarbonGuru MV3 Wrapper] Created timeout alarm ${alarmName} (${ms}ms)`);

    return id;
};

self.clearTimeout = (id) => {
    // Check for native timeout
    const nativeKey = `native_timeout_${id}`;
    if (timeoutCallbacks.has(nativeKey)) {
        const entry = timeoutCallbacks.get(nativeKey);
        nativeClearTimeout(entry.nativeId);
        timeoutCallbacks.delete(nativeKey);
        return;
    }

    // Check for alarm-based timeout
    const alarmName = `${TIMEOUT_PREFIX}${id}`;
    if (timeoutCallbacks.has(alarmName)) {
        chrome.alarms.clear(alarmName);
        timeoutCallbacks.delete(alarmName);
        console.log(`[CarbonGuru MV3 Wrapper] Cleared timeout alarm ${alarmName}`);
        return;
    }

    // Fallback: try clearing as native timeout ID directly
    nativeClearTimeout(id);
};

/**
 * Handle all alarm callbacks (both intervals and timeouts)
 */
chrome.alarms.onAlarm.addListener((alarm) => {
    // Handle timeout alarms
    if (alarm.name.startsWith(TIMEOUT_PREFIX)) {
        const entry = timeoutCallbacks.get(alarm.name);
        if (!entry) {
            chrome.alarms.clear(alarm.name);
            return;
        }

        try {
            entry.callback(...entry.args);
        } catch (error) {
            console.error(`[CarbonGuru MV3 Wrapper] Timeout callback error (${alarm.name}):`, error);
        }

        // Timeouts are one-shot - remove after execution
        timeoutCallbacks.delete(alarm.name);
        return;
    }

    // Handle interval alarms
    if (!alarm.name.startsWith(ALARM_PREFIX)) return;

    const entry = alarmCallbacks.get(alarm.name);
    if (!entry) {
        // Orphaned alarm - clear it
        chrome.alarms.clear(alarm.name);
        return;
    }

    try {
        entry.callback(...entry.args);
    } catch (error) {
        console.error(`[CarbonGuru MV3 Wrapper] Alarm callback error (${alarm.name}):`, error);
    }

    // For one-shot alarms (< 60s intervals), reschedule
    if (entry.ms < 60000) {
        scheduleOneShotAlarm(alarm.name, entry.ms);
    }
});

/**
 * 3. Service Worker lifecycle management
 */
self.addEventListener('install', (event) => {
    console.log('[CarbonGuru MV3 Wrapper] Service worker installed');
    self.skipWaiting();
});

self.addEventListener('activate', (event) => {
    console.log('[CarbonGuru MV3 Wrapper] Service worker activated');
    event.waitUntil(self.clients.claim());
});

/**
 * 4. Restore intervals on service worker startup
 *
 * When the SW restarts, we can't restore the actual callbacks (they're gone),
 * but we can log what was running and clear stale alarms.
 * The extension code will re-create necessary intervals on init().
 */
chrome.runtime.onStartup.addListener(async () => {
    console.log('[CarbonGuru MV3 Wrapper] Browser startup - checking for stale intervals');
    try {
        const data = await chrome.storage.local.get([STORAGE_KEY]);
        const intervals = data[STORAGE_KEY] || {};
        const staleCount = Object.keys(intervals).length;

        if (staleCount > 0) {
            console.log(`[CarbonGuru MV3 Wrapper] Found ${staleCount} persisted interval(s) from previous session`);
            // Clear stale alarms - extension will recreate what it needs
            for (const alarmName of Object.keys(intervals)) {
                await chrome.alarms.clear(alarmName);
            }
            await chrome.storage.local.remove([STORAGE_KEY]);
            console.log('[CarbonGuru MV3 Wrapper] Cleared stale intervals');
        }
    } catch (e) {
        console.warn('[CarbonGuru MV3 Wrapper] Error checking stale intervals:', e);
    }
});

/**
 * 5. Keep service worker alive during active operations
 *
 * When content scripts or popup connect via runtime.connect, we keep
 * the connection alive to prevent SW termination during active work.
 */
const activePorts = new Set();

chrome.runtime.onConnect.addListener((port) => {
    if (port.name === 'cg_keepalive') {
        activePorts.add(port);
        console.log(`[CarbonGuru MV3 Wrapper] Keepalive port connected (${activePorts.size} active)`);

        port.onDisconnect.addListener(() => {
            activePorts.delete(port);
            console.log(`[CarbonGuru MV3 Wrapper] Keepalive port disconnected (${activePorts.size} active)`);
        });
    }
});

// ==================== LOAD EXTENSION CODE ====================

console.log('[CarbonGuru MV3 Wrapper] Loading browser-polyfill and background.js...');

try {
    // Load browser polyfill first (provides Promise-based API)
    importScripts('browser-polyfill.min.js');
    console.log('[CarbonGuru MV3 Wrapper] browser-polyfill loaded');

    // Load oauth.js first (OAuthManager class used by background.js)
    importScripts('oauth.js');
    console.log('[CarbonGuru MV3 Wrapper] oauth.js loaded');

    // Load the actual extension code (same as Firefox uses)
    importScripts('background.js');
    console.log('[CarbonGuru MV3 Wrapper] background.js loaded successfully');

} catch (error) {
    console.error('[CarbonGuru MV3 Wrapper] Failed to load scripts:', error);
}

console.log('[CarbonGuru MV3 Wrapper] Initialization complete');
